*** mse-qrt-a.do
***
*** Calcuate MSE of quasi-real time forecasts of SSUR-2 and SSUR-3 using GB/SPF dates
***
clear all
set more off
set type double

global path = "$HOME/research/ssur"

*******************************
*** UPDATE THESE PARAMETERS ***
*******************************
* date of first and last forecast to evaluate
local start = tm(1976m1)
local stop = tm(2006m12)

*** V1 UR DATA
tempfile v1ur
use $path/data/rtdata/v1ur.dta, clear
replace v1ur=v1ur/100
tscollap v1ur, to(q) gen(qdate)
qui replace v1ur = round(v1ur,.0001)
tsset qdate, q
save `v1ur'

*** PROCESS GB FORECASTS
tempfile gbur
use $path/data/gb/gbur.dta, clear
reshape long ur_gb, i(gbdate) j(j)
keep if j<5
gen int fcdate=mofd(gbdate)
* fix multiple GBs per month
drop if gbdate==td(29apr1970)
qui replace fcdate=tm(1971m4) if gbdate==td(31mar1971)
qui replace fcdate=tm(1971m5) if gbdate==td(28apr1971)
drop if gbdate==td(02jun1971)
qui replace fcdate=tm(1980m2) if gbdate==td(30jan1980)
*
gen int qdate=qofd(gbdate)+j
qui replace ur_gb = ur_gb/100
drop if ur_gb==.
format fcdate %tm
tsset fcdate qdate, q
keep fcdate qdate ur_gb
qui replace ur_gb = round(ur_gb,.0001)
save `gbur', replace

*** PROCESS SPF FORECASTS
tempfile spfur
use $path/data/spf/spf-unemp.dta, clear
drop *avg*
reshape long ru_med, i(qdate) j(j)
rename qdate date
gen int fcdate=mofd(dofq(date))+1
gen int qdate=date+j
gen ur_spf = ru_med/100
drop if ur_spf==.
format fcdate %tm
tsset fcdate qdate, q
keep fcdate qdate ur_spf
qui replace ur_spf = round(ur_spf,.0001)
save `spfur', replace

*** IDENTIFY FORECASTS IN COMMON
merge 1:1 fcdate qdate using `gbur'
keep if fcdate>=`start' & fcdate<=`stop'
tab fcdate if _merge==3, matrow(FCDATES)
local N = rowsof(FCDATES)

*** QRT SSUR-2 FORECASTS
tempfile ssur2
use $path/analysis-bpea/models/qrt/ssur2/qrt-ssur2.dta, clear
tsset fcdate mdate, m
* collapse to quarterly
tscollap ur_ssur2, to(q) gen(qdate)
* keep horizon t+0 to t+4
keep if qdate>=qofd(dofm(fcdate)) & qdate<qofd(dofm(fcdate))+5
qui replace ur_ssur2 = round(ur_ssur2,.0001)
save `ssur2'

*** QRT SSUR-3 FORECASTS
tempfile ssur3
use $path/analysis-bpea/models/qrt/ssur3/qrt-ssur3.dta, clear
drop lfpr_ssur3
tsset fcdate mdate, m
* collapse to quarterly
tscollap ur_ssur3, to(q) gen(qdate)
* keep horizon t+0 to t+4
keep if qdate>=qofd(dofm(fcdate)) & qdate<qofd(dofm(fcdate))+5
qui replace ur_ssur3 = round(ur_ssur3,.0001)
save `ssur3'

*** MERGE IN FORECASTS
use `ssur2', clear
merge 1:1 fcdate qdate using `ssur3', nogen

tsset fcdate qdate, q
order fcdate qdate ur_*

keep if ur_ssur2!=. & fcdate>=`start'

foreach var in ssur2 ssur3 {
  rename ur_`var' ur_`var'_
}

reshape wide ur_ssur2_ ur_ssur3_, i(qdate) j(fcdate)

*** MERGE IN V1 UR DATA
merge m:1 qdate using `v1ur', nogen keep(1 3)

*** CALCULATE FORECAST RESIDUALS
forvalues i = 1/`N' {
  local fcdate = FCDATES[`i',1]
  foreach var in ssur2 ssur3  {
    gen res_`var'_`fcdate' = 100*(v1ur-ur_`var'_`fcdate')
  }
}
keep qdate res*
*save res.dta, replace

*** CALCULATE STATISTICS
*use res.dta, replace
reshape long res_ssur2_ res_ssur3_, i(qdate) j(fcdate)
drop if res_ssur2_==.
keep if fcdate>=`start' & fcdate<=`stop'
sort fcdate qdate
by fcdate: gen byte j = _n
replace j=j-1
drop qdate
order fcdate j
tsset j fcdate, m
format fcdate %tm
foreach var in ssur2 ssur3 {
  rename res_`var'_ res_`var'
  gen se_`var' = res_`var'^2
}
reshape long res_ se_, i(fcdate j) j(model) s
rename res_ res
rename se_ se
egen mse = mean(se), by(j model)
gen rmse = sqrt(mse)

*** REPORT
cap log close
log using mse-qrt-a.log, text replace
sum fcdate if se!=.
format se res rmse %6.4f
local dfcbeg = string(qofd(dofm(r(min))),"%tq")
local dfcend = string(qofd(dofm(r(max))),"%tq")
noi disp _n "FOREACST EVALUATION"
noi disp "Evaluated over: `dfcbeg' to `dfcend'"
noi disp "Evaluated relative to: v1"
* OBS
noi disp _n "Observations" _c
noi table j model, c(freq) cell(11)
** MISS
*noi disp _n "Average miss (percentage point)" _c
*noi table j model, c(mean res) cell(11)
* RMSE
noi disp _n "Root mean squared error (percentage point)" _c
noi table j model, c(mean rmse) cell(11) format(%3.2f)

cap log close

if "$S_CONSOLE"=="console" exit, STATA clear
